﻿//=============================================================================
// SsaoBlur.fx Frank Luna (C) 2011 Wszelkie prawa zastrzeżone.
//
// Dokonuje dwustronnego rozmycia mapy okluzji otoczenia z zachowaniem krawędzi. Zamiast z   
// shadera obliczeniowego korzystamy z shadera pikseli, aby uniknąć przełączenia z 
// trybu obliczeń do trybu renderowania. Pamięć podręczna tekstury rekompensuje częściowo
// brak pamięci współdzielonej. Mapa okluzji otoczenia wykorzystuje
// 16-bitowy format tekstury, dzięki czemu pamięć podręczna pomieści wiele tekseli.
//=============================================================================

cbuffer cbPerFrame
{
	float gTexelWidth;
	float gTexelHeight;
};

cbuffer cbSettings
{
	float gWeights[11] = 
	{
		0.05f, 0.05f, 0.1f, 0.1f, 0.1f, 0.2f, 0.1f, 0.1f, 0.1f, 0.05f, 0.05f
	};
};

cbuffer cbFixed
{
	static const int gBlurRadius = 5;
};
 
// Danych nienumerycznych nie można zapisywać w obiekcie cbuffer.
Texture2D gNormalDepthMap;
Texture2D gInputImage;
 
SamplerState samNormalDepth
{
	Filter = MIN_MAG_LINEAR_MIP_POINT;

	AddressU = CLAMP;
	AddressV = CLAMP;
};

SamplerState samInputImage
{
	Filter = MIN_MAG_LINEAR_MIP_POINT;

	AddressU  = CLAMP;
    AddressV  = CLAMP;
};

struct VertexIn
{
	float3 PosL    : POSITION;
	float3 NormalL : NORMAL;
	float2 Tex     : TEXCOORD;
};

struct VertexOut
{
    float4 PosH  : SV_POSITION;
	float2 Tex   : TEXCOORD;
};

VertexOut VS(VertexIn vin)
{
	VertexOut vout;
	
	// Już w przestrzeni NDC.
	vout.PosH = float4(vin.PosL, 1.0f);

	// Przekaż do shadera pikseli.
	vout.Tex = vin.Tex;
	
    return vout;
}


float4 PS(VertexOut pin, uniform bool gHorizontalBlur) : SV_Target
{
	float2 texOffset;
	if(gHorizontalBlur)
	{
		texOffset = float2(gTexelWidth, 0.0f);
	}
	else
	{
		texOffset = float2(0.0f, gTexelHeight);
	}

	// Wartość środkowa jest zawsze uwzględniana w sumie.
	float4 color      = gWeights[5]*gInputImage.SampleLevel(samInputImage, pin.Tex, 0.0);
	float totalWeight = gWeights[5];
	 
	float4 centerNormalDepth = gNormalDepthMap.SampleLevel(samNormalDepth, pin.Tex, 0.0f);

	for(float i = -gBlurRadius; i <=gBlurRadius; ++i)
	{
		// Środkową wagę dodaliśmy już wcześniej.
		if( i == 0 )
			continue;

		float2 tex = pin.Tex + i*texOffset;

		float4 neighborNormalDepth = gNormalDepthMap.SampleLevel(
			samNormalDepth, tex, 0.0f);

		//
		// Jeżeli wartość środkowa i sąsiednie wartości różnią się
		// za bardzo (wektorami normalnymi lub głębokością), przyjmujemy, że
		// mamy do czynienia z nieciągłym obszarem. Takie próbki wyłączamy z rozmycia.
		//
	
		if( dot(neighborNormalDepth.xyz, centerNormalDepth.xyz) >= 0.8f &&
		    abs(neighborNormalDepth.a - centerNormalDepth.a) <= 0.2f )
		{
			float weight = gWeights[i+gBlurRadius];

			// Dodaj sąsiedni piksel do rozmycia.
			color += weight*gInputImage.SampleLevel(
				samInputImage, tex, 0.0);
		
			totalWeight += weight;
		}
	}

	// Ponieważ część próbek została odrzucona, spraw, żeby wagi sumowały się do 1.
	return color / totalWeight;
}

technique11 HorzBlur
{
    pass P0
    {
		SetVertexShader( CompileShader( vs_5_0, VS() ) );
		SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS(true) ) );
    }
}

technique11 VertBlur
{
    pass P0
    {
		SetVertexShader( CompileShader( vs_5_0, VS() ) );
		SetGeometryShader( NULL );
        SetPixelShader( CompileShader( ps_5_0, PS(false) ) );
    }
}
 